/*****************************************************************************
 $Id: StegoAlgoRGB.java,v 1.1 2002/04/13 17:42:10 bastian Exp $
 Part of TRex, (c) 2001, 2002 Bastian Friedrich <bastian@bastian-friedrich.de>
 This software licensed under the GPL.
 *****************************************************************************/
package trex.Algo;

import java.awt.*;
import java.awt.image.*;
import javax.swing.*;

import trex.*;
import trex.Filter.*;

/**
 * This extension to standard {@link trex.Algo.StegoAlgo}s allows for an
 * easier implementation, if the subjacent algorithm only works "pixel-wise",
 * taking one pixel at a time and returning another.
 * In this case, a TRexFilterRGB must be returned as declared by the
 * {@link #getDecryptFilter} and {@link #getEncryptFilter}. Traversing the image
 * then is done by the filtering routines of the {@link java.awt}.
 */

public abstract class StegoAlgoRGB extends StegoAlgo {

  /**
   * Return the string contained in the ImageIcon.
   * @param img The image hiding the data.
   * @return The hidden {@link java.lang.String}.
   */
  public String getDecrypted(ImageIcon img) throws DecryptImpossibleException {
    TRexFilterRGB myFilter =
      getDecryptFilter(img.getIconWidth(),
                       img.getIconHeight());

    new ImageIcon(Toolkit.getDefaultToolkit().createImage(
      new FilteredImageSource(
        img.getImage().getSource(),
        myFilter
      )
    ));

    return myFilter.getData();
  }


  /**
   * Return an image hiding the data.
   * @param data Data to hide.
   * @param img Envelope image.
   * @return The {@link javax.swing.ImageIcon} hiding the data.
   */
  public ImageIcon getEncrypted(String data, ImageIcon img) {
    if (pictureLargeEnough(img, data))
      return new ImageIcon(
                   Toolkit.getDefaultToolkit().createImage(
                     new FilteredImageSource(
                       img.getImage().getSource(),
                       getEncryptFilter(data,
                                        img.getIconWidth(),
                                        img.getIconHeight())
                     )
                   )
                 );
    else
      return null;

  }

  /**
   * Return a decrypt filter for this algorithm.
   * @param w Picture's width
   * @param h Picture's height
   * @return The {@link java.awt.image.ImageFilter} that does all the work.
   */
  protected abstract TRexFilterRGB getDecryptFilter(int w, int h);

  /**
   * Return an encrypt filter for this algorithm.
   * @param data Data to hide while filtering
   * @param w Picture's width
   * @param h Picture's height
   * @return The {@link java.awt.image.ImageFilter} that does all the work.
   */
  protected abstract TRexFilterRGB getEncryptFilter(String data, int w, int h);

  /**
   * Return the default amplification of this algorithm for the combinedPanel.
   * @return Default amplification: 255.
   */
  public int defaultAmplification() { return 255; }

}