/*****************************************************************************
 $Id: StegoAlgo.java,v 1.16 2002/04/14 03:42:33 bastian Exp $
 Part of TRex, (c) 2001, 2002 Bastian Friedrich <bastian@bastian-friedrich.de>
 This software licensed under the GPL.
 *****************************************************************************/
package trex.Algo;

import javax.swing.*;
import java.awt.*;
import java.awt.image.*;

import trex.*;
import trex.Filter.*;

/**
 * This class defines the interface of steganography algorithms.
 * @author Bastian Friedrich <a href="mailto:bastian@bastian-friedrich.de">&lt;bastian@bastian-friedrich.de&gt;</a>
 * @version $Revision: 1.16 $
 */

public abstract class StegoAlgo {

  /**
   * Algorithm's current pass phrase.
   */
  protected String passPhrase;

  /* Abstract functions */
  /**
   * Return whether this algorithm has a config dialog.
   * @return Config dialog availability.
   */
  public abstract boolean hasConfigDialog();

  /**
   * Return whether this algorithm needs a pass phrase.
   * Note that no pass phrase dialog is shown when the algorithm
   * has a config dialog.
   * @return Pass phrase necessity.
   */
  public abstract boolean hasPassPhrase();

  /**
   * Return an info string about this algorithm.
   * Should contain the algorithm's name and may include additional information.
   * @return Info string.
   */
  public abstract String getInfo();

  /**
   * If hasConfigDialog() is true, return a panel with the config dialog.
   * Note that this dialog has to include a potential pass phrase request,
   * as <code>hasPassPhrase()</code> is ignored when there is a config dialog.
   * Should return <code>null</code> when there is no config dialog.
   * @see #hasConfigDialog
   * @see #hasPassPhrase
   * @return Panel containing config dialog.
   */
  public abstract java.awt.Component getConfigDialog();

  /**
   * Return whether the config from pass phrase or config dialog are valid.
   * @return Config validity.
   */
  public abstract boolean validConfig();

  /**
   * Set new pass phrase.
   * @param pp New pass phrase.
   */
  public void setPassPhrase(String pp) {
    passPhrase = pp;
  }

  /**
   * Return the string contained in the ImageIcon.
   * @param img The image hiding the data.
   * @return The hidden {@link java.lang.String}.
   */
  public abstract String getDecrypted(ImageIcon img) throws DecryptImpossibleException;

  /**
   * Return an image hiding the data.
   * @param data Data to hide.
   * @param img Envelope image.
   * @return The {@link javax.swing.ImageIcon} hiding the data.
   */
  public abstract ImageIcon getEncrypted(String data, ImageIcon img);

  /**
   * Return whether the passed picture is large enough to carry the data
   * @param img Image to hide data in
   * @param data data to hide
   * @return A boolean indicating whether the image is large enough for the data
   */
  public abstract boolean pictureLargeEnough(ImageIcon img, String data);

  /**
   * Return the default amplification of this algorithm for the combinedPanel.
   * @return Default amplification.
   */
  public abstract int defaultAmplification();
}
